using System;
using System.Xml;
using System.Reflection;
using System.Runtime.Remoting;

using Bertaccini.Utils;

namespace Team_Project
{
	/// <summary>
	/// Questa classe gestisce la catena dei messaggi essendo in grado di 
	/// identificare il prossimo anello della catena attraverso dei "token".
	/// </summary>
	/// <remarks><para>Nella sua versione prototipale il manager  stato utilizzato
	/// solo con catene lineari. E' per predisposto per gestire anche ramificazioni
	/// della catena.</para>
	/// <para>Il singleton  disponibile globalmente con il nome "ChainManager" dalla
	/// classe <see cref="Team_Project.Globals.Data">Global</see></para></remarks>
	[TeamProjectInit("ChainManager")]
	public class MsgElabChainManager : MarshalByRefObject
	{
		/// <summary>
		/// Inizializza un'istanza di MsgElabChainManager
		/// </summary>
		/// <remarks>Questa funzione  richiesta per l'inizializzazione
		/// effettuata dall'Engine.
		/// Utilizza il file MessageChain.xml
		/// </remarks>
		/// <returns>Un MsgElabChainManager configurato.</returns>
		public static MsgElabChainManager ComponentInit()
		{
			string cfgName = "Config\\MessageChain.xml";
#if DEBUG
			cfgName = @"..\..\..\" + cfgName;
#endif
			return new MsgElabChainManager(cfgName);
		}

		/// <summary>
		/// Radice della "catena"
		/// </summary>
		protected TreeElement chainRoot;
		/// <summary>
		/// Costruisce una nuova istanza del manager
		/// </summary>
		/// <param name="configPath">Percorso del file xml contenente
		/// le specifiche per la configurazione della catena</param>
		public MsgElabChainManager(string configPath)
		{
			XmlDocument doc = new XmlDocument();
			doc.Load(configPath);
			XmlNode root = doc.SelectSingleNode("/MessageChain/Ring");
			IOperationHandler oh = CreateOpHandler(root);
			chainRoot = new TreeElement(oh);
			BuildChildren(root,chainRoot);
		}
		/// <summary>
		/// Crea l'handler di messaggio (anello della catena) utilizzando i
		/// dati all'interno del nodo.
		/// </summary>
		/// <param name="nd">Elemento xml contenente i dati necessari per
		/// la creazione dell'handler</param>
		/// <returns>Handler (anello della catena) individuato dal nodo</returns>
		protected static IOperationHandler CreateOpHandler(XmlNode nd)
		{
			if(nd.Attributes["Engine"] != null) return Globals.Instance.Engine;
			string asmName = nd.Attributes["Assembly"].Value;
			Assembly asm = Assembly.Load(asmName);
			string typeName = nd.Attributes["Type"].Value;
			Type t = asm.GetType(typeName);
			return (IOperationHandler) Activator.CreateInstance(t);
		}

		/// <summary>
		/// Funzione ricorsiva utilizzata per la lettura dell'albero xml.
		/// Esplora i nodi "Ring" dell'albero e costruisce gli handler associati
		/// </summary>
		/// <param name="nd">Nodo xml contenente i dati per la creazione
		/// dell'handler. Verranno ricorsivamente esplorati tutti i figli "Ring"
		/// di questo nodo.</param>
		/// <param name="par">Elemento superiore della catena, che rappresenta
		/// i dati letti dal nodo al parametro nd. Gli handler creati dai figli
		/// del nodo nd verranno aggiunti come figli di par.
		/// </param>
		protected static void BuildChildren(XmlNode nd,TreeElement par)
		{
			foreach(XmlNode cn in nd.SelectNodes("Ring"))
			{
				TreeElement te = new TreeElement(CreateOpHandler(cn),par);
				BuildChildren(cn,te);
			}
		}

		/// <summary>
		/// Restrituisce la radice della catena dei messaggi.
		/// </summary>
		/// <returns>Il primo anello della catena.</returns>
		public IOperationHandler GetRoot()
		{
			IOperationHandler oh =(IOperationHandler)chainRoot.Content;
			oh.ChainToken = chainRoot;
			return oh;
		}

		/// <summary>
		/// Restituisce il prossimo nodo della catena, valutando la condizione.
		/// </summary>
		/// <param name="chainToken">Token dell'anello che effettua l'invocazione</param>
		/// <param name="conditionIndex">Identifica quale ramo dell'albero della "catena"
		/// dev'essere utilizzato. Attualmente l'indice corrisponde alla posizione
		/// del figlio nell'albero xml della configurazione.</param>
		/// <returns>Il prossimo anello della catena.</returns>
		public IOperationHandler GetNextLevel(object chainToken,int conditionIndex)
		{
			TreeElement curr = (TreeElement)chainToken;
			TreeElement next = curr[conditionIndex];
			IOperationHandler oh =(IOperationHandler)next.Content;
			oh.ChainToken = next;
			return oh;
		}

		/// <summary>
		/// Restituisce il prossimo nodo della catena considerando la catena lineare
		/// o restituendo comunque il gestore associato alla condizione 0
		/// </summary>
		/// <param name="chainToken">Token dell'anello che effettua l'invocazione</param>
		/// <returns>Il prossimo anello della catena.</returns>
		public IOperationHandler GetNextLevel(object chainToken)
		{return GetNextLevel(chainToken,0);}
	}
}
